using System;
using System.Data;
using System.Data.SqlClient;
using System.Reflection;
using gov.va.med.vbecs.ExceptionManagement;
using STOREDPROC = gov.va.med.vbecs.Common.VbecsStoredProcs;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using gov.va.med.vbecs.Common;

namespace gov.va.med.vbecs.DAL
{
		#region Header

		///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
		///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
		///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/11/2003</CreationDate>
		///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
		///<summary>
		///	Data access component implementing data persistence logic for a specimen.
		///	Allows retrive/save specimen data from/to database.
		///</summary>

		#endregion

	public class Specimen
	{
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4131"> 
		///		<ExpectedInput>Valid division code and specimen accession number</ExpectedInput>
		///		<ExpectedOutput>Data table with records</ExpectedOutput>>
		///	</Case>
		///
		///<Case type="1" testid ="4132"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Retrieves a specimen based on given division code and specimen accession number
		/// </summary>
		/// <param name="specimenUid">Specimen accession number</param>
		/// <returns>Single row DataTable containing a specimen data</returns>
		public static DataTable GetSpecimen(string specimenUid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSpecimen.divisioncode, System.Data.SqlDbType.VarChar),
				new SqlParameter(STOREDPROC.GetSpecimen.specimenuid, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = Common.LogonUser.LogonUserDivisionCode;
			prms[1].Value = specimenUid;

			DataSet	ds = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimen.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Carl Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2791"> 
		///		<ExpectedInput>Valid specimen GUID</ExpectedInput>
		///		<ExpectedOutput>Data table with records</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2792"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>Retrieves a specimen based on given division code and specimen accession number</summary>
		/// <param name="patientSpecimenGuid">Specimen GUID</param>
		/// <returns>Single row DataTable containing a specimen data</returns>
		public static DataTable GetSpecimen(Guid patientSpecimenGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetSpecimenByGuid.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetSpecimenByGuid.divisioncode, System.Data.SqlDbType.Char)
				};
			
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet	ds = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenByGuid.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2793"> 
		///		<ExpectedInput>Valid data tables</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8140"> 
		///		<ExpectedInput>Valid data tables including exception</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2794"> 
		///		<ExpectedInput>Invalid RowVersion</ExpectedInput>
		///		<ExpectedOutput>RowVersionException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///	Saves specimen and specimen comment data to the database from 
		///	two corresponding DataTables
		///</summary>
		///<param name="dtPatientSpecimen">DataTable containing specimen data to save</param>
		///<param name="dtExceptionSpecimen">DataTable containing exceptions</param>
		///<param name="lastUpdateFunctionId">UC or calling method</param>
		///<param name="dtWorkloadEvents">Workload Event data containing specimen GUID</param>
		public static int UpdateSpecimen(DataTable dtPatientSpecimen, DataTable dtExceptionSpecimen, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
            StoredProcedure spSpecimen = new StoredProcedure();
			int returnValue = 0;
			

			if (dtExceptionSpecimen == null)
			{
				DataTable[] dtArray = new DataTable[2];
				string[] sprocNames = new string[2];

				if(dtWorkloadEvents != null)
				{
					sprocNames[0] = (STOREDPROC.InsertWorkloadEvents.StoredProcName);
					dtArray[0] = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
                }

				sprocNames[1] = (STOREDPROC.UpdatePatientSpecimen.StoredProcName);
				dtArray[1] = Common.Utility.AppendLastUpdateInformation(dtPatientSpecimen,lastUpdateFunctionId);
										  
				returnValue = spSpecimen.TransactionalGetValue(sprocNames, dtArray);

			}
			else
			{
				DataTable[] dtArray = new DataTable[3];
				string[] sprocNames = new string[3];
				if(dtWorkloadEvents != null)
				{
					sprocNames[0] = (STOREDPROC.InsertWorkloadEvents.StoredProcName);
					dtArray[0] = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
				}

				sprocNames[1] = STOREDPROC.UpdatePatientSpecimen.StoredProcName;
				dtArray[1] = Common.Utility.AppendLastUpdateInformation(dtPatientSpecimen,lastUpdateFunctionId);

				sprocNames[2] = STOREDPROC.InsertExceptionSpecimen.StoredProcName;
				dtArray[2] = dtExceptionSpecimen;

				
				returnValue = spSpecimen.TransactionalGetValue(sprocNames, dtArray);
			}
			
			return returnValue;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2789"> 
		///		<ExpectedInput>DataTables</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="8139"> 
		///		<ExpectedInput>Valid datatables with exception</ExpectedInput>
		///		<ExpectedOutput>0</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2790"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// InsertSpecimen
		/// </summary>
		/// <param name="dtPatientSpecimen"></param>
		/// <param name="dtExceptionSpecimen"></param>
		/// <param name="lastUpdateFunctionId"></param>
		/// <param name="dtWorkloadEvents"></param>
		/// <returns></returns>
		public static int InsertSpecimen(DataTable dtPatientSpecimen, DataTable dtExceptionSpecimen, DataTable dtWorkloadEvents, Common.UpdateFunction lastUpdateFunctionId)
		{
			System.Collections.ArrayList sprocArray = new System.Collections.ArrayList();
			System.Collections.ArrayList dtArray = new System.Collections.ArrayList();

			dtArray.Add(Common.Utility.AppendLastUpdateInformation(dtPatientSpecimen,lastUpdateFunctionId));

			sprocArray.Add(STOREDPROC.InsertPatientSpecimen.StoredProcName);

			if (dtExceptionSpecimen != null)
			{
				dtArray.Add(dtExceptionSpecimen);
				sprocArray.Add(STOREDPROC.InsertExceptionSpecimen.StoredProcName);
			}

			if(dtWorkloadEvents != null)
			{
				dtWorkloadEvents = Common.Utility.AppendLastUpdateInformation(dtWorkloadEvents,lastUpdateFunctionId);
				dtArray.Add(dtWorkloadEvents);
				sprocArray.Add(STOREDPROC.InsertWorkloadEvents.StoredProcName);
			}

			int	returnValue = new Common.StoredProcedure().TransactionalGetValue(sprocArray, dtArray);
			
			return returnValue;
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1401"> 
		///		<ExpectedInput>Guid associated with diagnostic test</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1402"> 
		///		<ExpectedInput>Guid associated with diagnostic test</ExpectedInput>
		///		<ExpectedOutput>false</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///	Checks if specimen with a given GUID is associated with ABO/Rh, DAT, 
		///	ABS, or Transfusion Reaction Workup
		///</summary>
		///<param name="patientSpecimenGuid">Specimen GUID</param>
		///<returns>Boolean check result</returns>
		public static bool IsSpecimenAssociatedWithDiagnosticTest (Guid patientSpecimenGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.IsSpecimenAssociatedWithDiagnosticTest.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.IsSpecimenAssociatedWithDiagnosticTest.divisioncode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = patientSpecimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			return new StoredProcedure().GetBoolReturnValue( STOREDPROC.IsSpecimenAssociatedWithDiagnosticTest.StoredProcName, prms );
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2787"> 
		///		<ExpectedInput>Valid accession # and SSN</ExpectedInput>
		///		<ExpectedOutput>Data table with records</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2788"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///<summary>
		///	Returns information to fill order search screen
		///</summary>
		///<param name="specimenUid">Specimen UID</param>
		///<param name="patientSsn">Patient SSN</param>
		///<param name="patientLastName">Patient Name</param>
		///<returns>DataTable with specimen details</returns>
		public static DataTable GetSpecimenSearchDetails(string specimenUid, string patientSsn, string patientLastName)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetSpecimenSearchDetails.specimenuid, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenSearchDetails.patientssn, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenSearchDetails.patientlastname, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenSearchDetails.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = (specimenUid.Length == 0) ? null : specimenUid;
			prms[1].Value = (patientSsn.Length == 0) ? null : patientSsn;			
			prms[2].Value = (patientLastName.Length == 0) ? null : patientLastName;
			prms[3].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenSearchDetails.StoredProcName, prms);

			return ds.Tables[0];
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/2/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3504"> 
		///		<ExpectedInput>specimenUid, patientSSn, PatientLastName</ExpectedInput>
		///		<ExpectedOutput>Data table</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3505"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>
		///	Returns information to fill specimen search screen
		///</summary>
		///<param name="specimenUid">Specimen UID</param>
		///<param name="patientSsn">Patient SSN</param>
		///<param name="patientLastName">Patient Name</param>
		///<param name="expiresAfter"></param>
		///<returns>DataTable with specimen details</returns>
		public static DataTable GetSpecimenList(string specimenUid, string patientSsn, string patientLastName, DateTime expiresAfter)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetSpecimenList.specimenuid, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenList.patientssn, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenList.patientlastname, System.Data.SqlDbType.VarChar),
					new SqlParameter(STOREDPROC.GetSpecimenList.specimenexpirationdate, System.Data.SqlDbType.DateTime),
					new SqlParameter(STOREDPROC.GetSpecimenList.divisioncode, System.Data.SqlDbType.VarChar)
				};

			prms[0].Value = (specimenUid.Length == 0) ? null : specimenUid;
			prms[1].Value = (patientSsn.Length == 0) ? null : patientSsn;			
			prms[2].Value = (patientLastName.Length == 0) ? null : patientLastName;
			prms[3].Value = expiresAfter;
			prms[4].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenList.StoredProcName, prms);

			return ds.Tables[0];
		}


		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3502"> 
		///		<ExpectedInput>Valid GUID</ExpectedInput>
		///		<ExpectedOutput>DataTable w/rows</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3503"> 
		///		<ExpectedInput>Invalid Guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetTestResultsForSpecimen
		/// </summary>
		/// <param name="patientSpecimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetTestResultsForSpecimen(Guid patientSpecimenGuid)
		{
			SqlParameter[] prms =
				{
					new SqlParameter(STOREDPROC.GetTestResultsForSpecimen.patientspecimenguid, System.Data.SqlDbType.UniqueIdentifier),
					new SqlParameter(STOREDPROC.GetTestResultsForSpecimen.divisioncode, System.Data.SqlDbType.Char)
				};

			prms[0].Value = patientSpecimenGuid;			
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet ds = Common.StoredProcedure.GetData(STOREDPROC.GetTestResultsForSpecimen.StoredProcName, prms);

			return ds.Tables[0];
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1397"> 
		///		<ExpectedInput>true</ExpectedInput>
		///		<ExpectedOutput>DataTable without RowVersion</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1398"> 
		///		<ExpectedInput>false</ExpectedInput>
		///		<ExpectedOutput>DataTable with RowVersion</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetEmptyPatientSpecimenTable
		/// </summary>
		/// <param name="isNew">New specimen indicator</param>
		/// <returns></returns>
		public static DataTable GetEmptyPatientSpecimenTable(bool isNew)
		{
			DataTable dtEmpty = new DataTable(TABLE.PatientSpecimen.TableName);

			dtEmpty.Columns.Add(TABLE.PatientSpecimen.DivisionCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.PatientGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.PatientSpecimenGuid, typeof(System.Guid));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.PhlebotomistId, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.PostTransfusionIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenAccessionNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenCollectionDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenExpirationDate, typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenReceivedDate , typeof(System.DateTime));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenSource, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenStatusCode, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenUid, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.UnacceptableSpecimenIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.VbecsSpecimenNumber, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.EntryTechId, typeof(string));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.MaintainableSpecimenIndicator, typeof(bool));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.CannedCommentGuid, typeof(Guid));
			dtEmpty.Columns.Add(TABLE.PatientSpecimen.SpecimenComments, typeof(string));
			if (!isNew)
			{
				dtEmpty.Columns.Add(TABLE.PatientSpecimen.RowVersion, typeof(byte[]));
			}

			return dtEmpty;
		}

		///<Developers>
		///	<Developer>C. Jensen</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/16/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5428"> 
		///		<ExpectedInput>Specimen Guid</ExpectedInput>
		///		<ExpectedOutput>DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5429"> 
		///		<ExpectedInput>Invalid guid</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// GetSpecimenHistory
		/// </summary>
		/// <param name="specimenGuid"></param>
		/// <returns></returns>
		public static DataTable GetSpecimenHistory(Guid specimenGuid)
		{
			SqlParameter[] prms =
			{
				new SqlParameter(STOREDPROC.GetSpecimenHistory.PatientSpecimenGuid, System.Data.SqlDbType.UniqueIdentifier),
				new SqlParameter(STOREDPROC.GetSpecimenHistory.DivisionCode, System.Data.SqlDbType.VarChar)
			};
			
			prms[0].Value = specimenGuid;
			prms[1].Value = Common.LogonUser.LogonUserDivisionCode;

			DataSet	ds = Common.StoredProcedure.GetData(STOREDPROC.GetSpecimenHistory.StoredProcName, prms);

			return ds.Tables[0];
		}

	}
}
